﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using gov.va.med.vbecs.ExceptionManagement;
using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.Lock
{
    public abstract class SyncLockBase : IDisposable
    {
        private const int GlobalSyncLockWaitTimeoutMs = 1000;
		
		/// <summary>
		/// Global access synchronization token for global mutex used to ensure 
		/// exclusive access to global mutexes when creating/checking locks.
		/// </summary>
		protected const string GlobalMutexAccessSyncToken = @"Global\gov.va.med.vbecs.GlobalAppLockSyncToken";

		/// <summary>
		/// Global VBECS application synchronization tokens (mutex name).
		/// Used to flag running VBECS application on the server. 
		/// Two are needed; one for test and another for prod.
		/// </summary>
		protected const string GlobalTestVbecsSyncToken = @"Global\gov.va.med.vbecs.GlobalTestVbecsSyncToken";

		/// <summary>
		/// GlobalProdVbecsSyncToken
		/// </summary>
		protected const string GlobalProdVbecsSyncToken = @"Global\gov.va.med.vbecs.GlobalProdVbecsSyncToken";
		
		/// <summary>
		/// Global VBECS Administrator application single user mode sychronization tokens (mutex name). 
		/// Used to flag VBECS Administrator application running single user mode.
		/// Two are needed; one for test and another for prod.
		/// </summary>
		protected const string GlobalAdminTestSingleUserModeSyncToken = @"Global\gov.va.med.vbecs.GlobalAdminTestSingleUserModeSyncToken";

		/// <summary>
		/// GlobalAdminProdSingleUserModeSyncToken
		/// </summary>
		protected const string GlobalAdminProdSingleUserModeSyncToken = @"Global\gov.va.med.vbecs.GlobalAdminProdSingleUserModeSyncToken";
 
		/// <summary>
		/// Base constructor for VBECS and Administrator application locks. 
		/// Obtains exclusive serverwide lock, creates/checks application specific locks 
		/// defined by the derived classes and then releases exclusive lock. 
		/// </summary>
		protected SyncLockBase()
		{
            Init();
            using (obtainGlobalLock())
            {
                ObtainAppSpecificGlobalLocks();
            }
		}

        /// <summary>
        /// Perform innitialization here is necessary. 
        /// Happens before global locks obtained
        /// </summary>
        protected virtual void Init()
        { }

		/// <summary>
		/// Creates/checks application-specific locks defined by derived classes. 
		/// </summary>
		protected abstract void ObtainAppSpecificGlobalLocks();

		/// <summary>
		/// Releases application-specific locks defined by derived classes. 
		/// </summary>
		protected abstract void ReleaseAppSpecificGlobalLocks();

		/// <summary>
		/// Obtains global synchronization lock to ensure exclusive access when creating/checking locks.
		/// </summary>
        private SharedMutex obtainGlobalLock()
		{
            SharedMutex m = new SharedMutex(GlobalMutexAccessSyncToken);
            try
            {
                m.Obtain(GlobalSyncLockWaitTimeoutMs);
            }
            catch (TimeoutException ex)
            {
                throw (new BaseApplicationException(StrRes.SysErrMsg.Common.TimedOutWhileWaitingForGlobalSyncLock().ResString, ex));
            }
            return m;
		}

		/// <summary>
		/// Finalizes the instance.
		/// </summary>
		~SyncLockBase()
		{
			Dispose( false );
		}

		/// <summary>
		/// Disposes the object releasing all locks and performing the cleanup.
		/// </summary>
		public void Dispose()
		{
			GC.SuppressFinalize( this );
			Dispose( true );
		}

		private void Dispose( bool disposing )
		{				
			lock( this )
			{
				if( disposing )
				{
                    using (obtainGlobalLock())
                    {
                        ReleaseAppSpecificGlobalLocks();
                    }
				}
			}
		}

    }
}
